import 'dotenv/config'
import { PrismaClient, UserRole, VehicleStatus, SyndicationStatus, VerificationStatus, InventoryMode, DealerSector } from '@prisma/client'
import { PrismaPg } from '@prisma/adapter-pg'
import { Pool } from 'pg'
import * as bcrypt from 'bcryptjs'

// Extract the direct connection URL from the Prisma Postgres URL
function getDirectUrl() {
  const url = process.env.DATABASE_URL
  if (!url) throw new Error('DATABASE_URL is not set')

  if (url.startsWith('postgres://') || url.startsWith('postgresql://')) {
    return url
  }

  if (url.startsWith('prisma+postgres://')) {
    const apiKeyMatch = url.match(/api_key=([^&]+)/)
    if (apiKeyMatch) {
      const decoded = JSON.parse(Buffer.from(apiKeyMatch[1], 'base64').toString())
      return decoded.databaseUrl
    }
  }

  return url
}

const directUrl = getDirectUrl()
const pool = new Pool({ connectionString: directUrl })
const adapter = new PrismaPg(pool)
const prisma = new PrismaClient({ adapter })

async function main() {
  console.log('Seeding database...')

  // Create Platform Admin
  const adminPassword = await bcrypt.hash('admin123', 10)
  const admin = await prisma.user.upsert({
    where: { email: 'admin@dms.co.nz' },
    update: {},
    create: {
      email: 'admin@dms.co.nz',
      passwordHash: adminPassword,
      name: 'Platform Admin',
      role: UserRole.platform_admin,
      isActive: true
    }
  })
  console.log('Created admin user:', admin.email)

  // Create Test Dealers (Vehicles Sector)
  const dealers = [
    {
      name: 'Auckland Motors Ltd',
      contactEmail: 'sales@aucklandmotors.co.nz',
      contactPhone: '09 555 1234',
      address: '123 Queen Street, Auckland CBD',
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date(),
      sector: DealerSector.vehicles
    },
    {
      name: 'Wellington Auto Group',
      contactEmail: 'info@wellingtonauto.co.nz',
      contactPhone: '04 555 5678',
      address: '456 Lambton Quay, Wellington',
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date(),
      sector: DealerSector.vehicles
    },
    {
      name: 'Christchurch Cars',
      contactEmail: 'contact@chchcars.co.nz',
      contactPhone: '03 555 9012',
      address: '789 Colombo Street, Christchurch',
      verificationStatus: VerificationStatus.pending,
      sector: DealerSector.vehicles
    },
    {
      name: 'Tristram Auckland',
      contactEmail: 'sales@tristramauckland.co.nz',
      contactPhone: '09 525 5039',
      address: '501 Great South Road, Greenlane, Auckland',
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date(),
      sector: DealerSector.vehicles
    },
    {
      name: 'Tristram Hamilton',
      contactEmail: 'sales@tristramhamilton.co.nz',
      contactPhone: '07 839 0500',
      address: '100 Te Rapa Road, Hamilton',
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date(),
      sector: DealerSector.vehicles
    },
    {
      name: 'Tristram Tauranga',
      contactEmail: 'sales@tristramtauranga.co.nz',
      contactPhone: '07 578 0099',
      address: '88 Cameron Road, Tauranga',
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date(),
      sector: DealerSector.vehicles
    }
  ]

  const createdDealers = []
  for (const dealer of dealers) {
    const created = await prisma.dealer.upsert({
      where: { id: dealer.name.toLowerCase().replace(/\s+/g, '-') },
      update: dealer,
      create: {
        id: dealer.name.toLowerCase().replace(/\s+/g, '-'),
        ...dealer
      }
    })
    createdDealers.push(created)
    console.log('Created dealer:', created.name)
  }

  // Create Dealer Group (Tristram Group)
  const tristramGroup = await prisma.dealerGroup.upsert({
    where: { id: 'tristram-group' },
    update: {},
    create: {
      id: 'tristram-group',
      name: 'Tristram Group',
      contactEmail: 'admin@tristramgroup.co.nz',
      contactPhone: '09 525 5000',
      address: '501 Great South Road, Greenlane, Auckland',
      inventoryMode: InventoryMode.separate,
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date()
    }
  })
  console.log('Created dealer group:', tristramGroup.name)

  // Link Tristram dealers to the group
  const tristramDealers = createdDealers.filter(d => d.name.startsWith('Tristram'))
  for (const dealer of tristramDealers) {
    await prisma.dealer.update({
      where: { id: dealer.id },
      data: { groupId: tristramGroup.id }
    })
    console.log('  Linked', dealer.name, 'to', tristramGroup.name)
  }

  // Create Dealer Users
  const userPassword = await bcrypt.hash('password123', 10)
  const dealerUsers = [
    {
      email: 'john@aucklandmotors.co.nz',
      name: 'John Smith',
      role: UserRole.dealer_admin,
      dealerId: createdDealers[0].id
    },
    {
      email: 'sarah@aucklandmotors.co.nz',
      name: 'Sarah Jones',
      role: UserRole.dealer_user,
      dealerId: createdDealers[0].id
    },
    {
      email: 'mike@wellingtonauto.co.nz',
      name: 'Mike Wilson',
      role: UserRole.dealer_admin,
      dealerId: createdDealers[1].id
    },
    {
      email: 'lisa@chchcars.co.nz',
      name: 'Lisa Brown',
      role: UserRole.dealer_admin,
      dealerId: createdDealers[2].id
    },
    {
      email: 'sales@tristramauckland.co.nz',
      name: 'Tristram Sales',
      role: UserRole.dealer_admin,
      dealerId: createdDealers[3].id
    }
  ]

  for (const user of dealerUsers) {
    const created = await prisma.user.upsert({
      where: { email: user.email },
      update: {},
      create: {
        ...user,
        passwordHash: userPassword,
        isActive: true
      }
    })
    console.log('Created user:', created.email)
  }

  // Create Group Admin user for Tristram Group
  const groupAdmin = await prisma.user.upsert({
    where: { email: 'admin@tristramgroup.co.nz' },
    update: {},
    create: {
      email: 'admin@tristramgroup.co.nz',
      passwordHash: userPassword,
      name: 'Tristram Group Admin',
      role: UserRole.group_admin,
      isActive: true
    }
  })
  console.log('Created group admin user:', groupAdmin.email)

  // Link group admin to Tristram Group
  await prisma.dealerGroupUser.upsert({
    where: {
      userId_groupId: { userId: groupAdmin.id, groupId: tristramGroup.id }
    },
    update: {},
    create: {
      userId: groupAdmin.id,
      groupId: tristramGroup.id
    }
  })
  console.log('  Linked', groupAdmin.email, 'to', tristramGroup.name)

  // Create IP Whitelist entries
  const ipEntries = [
    { ipAddress: '127.0.0.1', label: 'Localhost', dealerId: null as string | null, createdBy: admin.id },
    { ipAddress: '192.168.1.0/24', label: 'Auckland Motors Office', dealerId: createdDealers[0].id, createdBy: admin.id },
    { ipAddress: '10.0.0.0/24', label: 'Wellington Auto Office', dealerId: createdDealers[1].id, createdBy: admin.id }
  ]

  for (const entry of ipEntries) {
    // Handle null dealerId separately since composite unique can't use null in where clause
    const existing = await prisma.ipWhitelist.findFirst({
      where: {
        ipAddress: entry.ipAddress,
        dealerId: entry.dealerId
      }
    })
    if (!existing) {
      await prisma.ipWhitelist.create({ data: entry })
    }
  }
  console.log('Created IP whitelist entries')

  // Create Sample Vehicles
  const vehicles = [
    {
      dealerId: createdDealers[0].id,
      registration: 'ABC123',
      vin: 'JN1TBNT30Z0000001',
      make: 'Toyota',
      model: 'Corolla',
      year: 2021,
      engineSize: '1.8L',
      fuelType: 'Petrol',
      transmission: 'Automatic',
      bodyType: 'Hatchback',
      colour: 'Silver',
      mileage: 45000,
      price: 2499000,
      description: 'Well-maintained Toyota Corolla with full service history.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      nztaDataUsed: true
    },
    {
      dealerId: createdDealers[0].id,
      registration: 'XYZ789',
      vin: 'JN1TBNT30Z0000002',
      make: 'Ford',
      model: 'Ranger',
      year: 2022,
      engineSize: '2.0L',
      fuelType: 'Diesel',
      transmission: 'Automatic',
      bodyType: 'Ute',
      colour: 'White',
      mileage: 35000,
      price: 4500000,
      description: 'Ford Ranger XLT with tow bar and canopy.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      nztaDataUsed: true,
      wholesaleAvailable: true,
      wholesalePrice: 4200000,
      wholesaleListedAt: new Date()
    },
    {
      dealerId: createdDealers[0].id,
      registration: 'DEF456',
      make: 'Suzuki',
      model: 'Swift',
      year: 2020,
      engineSize: '1.2L',
      fuelType: 'Petrol',
      transmission: 'Automatic',
      bodyType: 'Hatchback',
      colour: 'Red',
      mileage: 55000,
      price: 1599000,
      status: VehicleStatus.sold,
      syndicationStatus: SyndicationStatus.none,
      nztaDataUsed: false
    },
    {
      dealerId: createdDealers[1].id,
      registration: 'WLG001',
      make: 'Mazda',
      model: 'CX-5',
      year: 2023,
      engineSize: '2.5L',
      fuelType: 'Petrol',
      transmission: 'Automatic',
      bodyType: 'SUV',
      colour: 'Blue',
      mileage: 15000,
      price: 3899000,
      description: 'Nearly new Mazda CX-5 with low kilometers.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      nztaDataUsed: true,
      wholesaleAvailable: true,
      wholesalePrice: 3600000,
      wholesaleListedAt: new Date()
    },
    {
      dealerId: createdDealers[1].id,
      registration: 'WLG002',
      make: 'Honda',
      model: 'Jazz',
      year: 2019,
      engineSize: '1.5L',
      fuelType: 'Hybrid',
      transmission: 'CVT',
      bodyType: 'Hatchback',
      colour: 'Black',
      mileage: 70000,
      price: 1899000,
      status: VehicleStatus.reserved,
      syndicationStatus: SyndicationStatus.synced,
      nztaDataUsed: true
    },
    {
      dealerId: createdDealers[2].id,
      registration: 'CHC001',
      make: 'Nissan',
      model: 'Leaf',
      year: 2022,
      engineSize: 'Electric',
      fuelType: 'Electric',
      transmission: 'Automatic',
      bodyType: 'Hatchback',
      colour: 'White',
      mileage: 25000,
      price: 2999000,
      description: 'Nissan Leaf with excellent range. Perfect city car.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.pending,
      nztaDataUsed: true
    },
    // Tristram Auckland vehicles
    {
      dealerId: createdDealers[3].id,
      registration: 'TRS001',
      make: 'BMW',
      model: '320i',
      year: 2022,
      engineSize: '2.0L',
      fuelType: 'Petrol',
      transmission: 'Automatic',
      bodyType: 'Sedan',
      colour: 'Black',
      mileage: 28000,
      price: 5499000,
      description: 'Elegant BMW 320i with M Sport package.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      lastSyncedAt: new Date(),
      nztaDataUsed: true
    },
    {
      dealerId: createdDealers[3].id,
      registration: 'TRS002',
      make: 'Mercedes-Benz',
      model: 'C200',
      year: 2021,
      engineSize: '2.0L',
      fuelType: 'Petrol',
      transmission: 'Automatic',
      bodyType: 'Sedan',
      colour: 'Silver',
      mileage: 35000,
      price: 4899000,
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      lastSyncedAt: new Date(),
      nztaDataUsed: true
    },
    // Tristram Hamilton vehicles
    {
      dealerId: createdDealers[4].id,
      registration: 'TRH001',
      make: 'Audi',
      model: 'A4',
      year: 2023,
      engineSize: '2.0L',
      fuelType: 'Petrol',
      transmission: 'Automatic',
      bodyType: 'Sedan',
      colour: 'White',
      mileage: 15000,
      price: 6299000,
      description: 'Nearly new Audi A4 with full warranty.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      lastSyncedAt: new Date(),
      nztaDataUsed: true
    },
    {
      dealerId: createdDealers[4].id,
      registration: 'TRH002',
      make: 'Volkswagen',
      model: 'Golf R',
      year: 2022,
      engineSize: '2.0L',
      fuelType: 'Petrol',
      transmission: 'DSG',
      bodyType: 'Hatchback',
      colour: 'Blue',
      mileage: 22000,
      price: 5199000,
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      lastSyncedAt: new Date(),
      nztaDataUsed: true
    },
    {
      dealerId: createdDealers[4].id,
      registration: 'TRH003',
      make: 'Porsche',
      model: 'Cayenne',
      year: 2021,
      engineSize: '3.0L',
      fuelType: 'Petrol',
      transmission: 'Automatic',
      bodyType: 'SUV',
      colour: 'Grey',
      mileage: 45000,
      price: 9999000,
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      lastSyncedAt: new Date(),
      nztaDataUsed: true
    },
    // Tristram Tauranga vehicles
    {
      dealerId: createdDealers[5].id,
      registration: 'TRT001',
      make: 'Lexus',
      model: 'RX350',
      year: 2022,
      engineSize: '3.5L',
      fuelType: 'Petrol',
      transmission: 'Automatic',
      bodyType: 'SUV',
      colour: 'Pearl White',
      mileage: 30000,
      price: 7499000,
      description: 'Luxury Lexus RX350 with premium package.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      lastSyncedAt: new Date(),
      nztaDataUsed: true
    },
    {
      dealerId: createdDealers[5].id,
      registration: 'TRT002',
      make: 'Land Rover',
      model: 'Range Rover Sport',
      year: 2021,
      engineSize: '3.0L',
      fuelType: 'Diesel',
      transmission: 'Automatic',
      bodyType: 'SUV',
      colour: 'Green',
      mileage: 40000,
      price: 8999000,
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      lastSyncedAt: new Date(),
      nztaDataUsed: true
    }
  ]

  for (const vehicle of vehicles) {
    await prisma.vehicle.create({
      data: vehicle
    })
  }
  console.log('Created', vehicles.length, 'sample vehicles')

  // Create NZTA cache entries
  const nztaRecords = [
    { registration: 'ABC123', make: 'Toyota', model: 'Corolla', year: 2021, engineSize: '1.8L', fuelType: 'Petrol', bodyType: 'Hatchback', colour: 'Silver' },
    { registration: 'XYZ789', make: 'Ford', model: 'Ranger', year: 2022, engineSize: '2.0L', fuelType: 'Diesel', bodyType: 'Ute', colour: 'White' },
    { registration: 'WLG001', make: 'Mazda', model: 'CX-5', year: 2023, engineSize: '2.5L', fuelType: 'Petrol', bodyType: 'SUV', colour: 'Blue' }
  ]

  for (const record of nztaRecords) {
    await prisma.nztaVehicle.upsert({
      where: { registration: record.registration },
      update: record,
      create: record
    })
  }
  console.log('Created NZTA cache entries')

  // ============================================
  // AGRICULTURE SECTOR
  // ============================================
  console.log('\n--- Creating Agriculture Sector Data ---')

  // Create Agriculture Dealers
  const agricultureDealers = [
    {
      name: 'Power Farming Auckland',
      contactEmail: 'sales@powerfarming-auckland.co.nz',
      contactPhone: '09 555 8000',
      address: '100 Great South Road, Penrose, Auckland',
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date(),
      sector: DealerSector.agriculture
    },
    {
      name: 'Central Tractors Hamilton',
      contactEmail: 'info@centraltractors.co.nz',
      contactPhone: '07 555 3000',
      address: '45 Te Rapa Road, Hamilton',
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date(),
      sector: DealerSector.agriculture
    },
    {
      name: 'Southern Farm Equipment',
      contactEmail: 'sales@southernfarm.co.nz',
      contactPhone: '03 555 4000',
      address: '200 Main Street, Ashburton',
      verificationStatus: VerificationStatus.verified,
      verifiedAt: new Date(),
      sector: DealerSector.agriculture
    }
  ]

  const createdAgDealers = []
  for (const dealer of agricultureDealers) {
    const created = await prisma.dealer.upsert({
      where: { id: dealer.name.toLowerCase().replace(/\s+/g, '-') },
      update: dealer,
      create: {
        id: dealer.name.toLowerCase().replace(/\s+/g, '-'),
        ...dealer
      }
    })
    createdAgDealers.push(created)
    console.log('Created agriculture dealer:', created.name)
  }

  // Create Agriculture Dealer Users
  const agDealerUsers = [
    {
      email: 'tom@powerfarming-auckland.co.nz',
      name: 'Tom Farmer',
      role: UserRole.dealer_admin,
      dealerId: createdAgDealers[0].id
    },
    {
      email: 'jim@centraltractors.co.nz',
      name: 'Jim Deere',
      role: UserRole.dealer_admin,
      dealerId: createdAgDealers[1].id
    },
    {
      email: 'bob@southernfarm.co.nz',
      name: 'Bob Harvester',
      role: UserRole.dealer_admin,
      dealerId: createdAgDealers[2].id
    }
  ]

  for (const user of agDealerUsers) {
    const created = await prisma.user.upsert({
      where: { email: user.email },
      update: {},
      create: {
        ...user,
        passwordHash: userPassword,
        isActive: true
      }
    })
    console.log('Created agriculture user:', created.email)
  }

  // Create Sample Equipment
  const equipment = [
    // Power Farming Auckland
    {
      dealerId: createdAgDealers[0].id,
      type: 'Tractor',
      brand: 'John Deere',
      model: '6130R',
      year: 2022,
      serialNumber: '1LV6130RHJJ123456',
      horsepower: 130,
      engineType: 'Diesel',
      hours: 1500,
      ptoType: '540/1000',
      hydraulicFlow: 114,
      liftCapacity: 5500,
      condition: 'Excellent',
      price: 18500000, // $185,000
      wholesalePrice: 17500000,
      description: 'Premium John Deere 6130R with CommandQuad transmission. Front loader ready.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      wholesaleAvailable: true,
      wholesaleListedAt: new Date()
    },
    {
      dealerId: createdAgDealers[0].id,
      type: 'Tractor',
      brand: 'Fendt',
      model: '724 Vario',
      year: 2021,
      serialNumber: 'FWAG724VHJJ789012',
      horsepower: 240,
      engineType: 'Diesel',
      hours: 2800,
      ptoType: '1000',
      hydraulicFlow: 200,
      liftCapacity: 10500,
      condition: 'Good',
      price: 32500000, // $325,000
      description: 'Fendt 724 Vario with Variogrip and full GPS guidance system.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced
    },
    {
      dealerId: createdAgDealers[0].id,
      type: 'Telehandler',
      brand: 'JCB',
      model: '541-70',
      year: 2020,
      serialNumber: 'JCB54170HJK345678',
      horsepower: 130,
      engineType: 'Diesel',
      hours: 3200,
      liftCapacity: 4100,
      condition: 'Good',
      price: 9500000, // $95,000
      description: 'JCB 541-70 Agri Super with 7m reach. Includes pallet forks and bucket.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced
    },
    // Central Tractors Hamilton
    {
      dealerId: createdAgDealers[1].id,
      type: 'Combine Harvester',
      brand: 'New Holland',
      model: 'CX8090',
      year: 2019,
      serialNumber: 'NHCX8090HJK567890',
      horsepower: 450,
      engineType: 'Diesel',
      hours: 1800,
      width: 9000,
      condition: 'Excellent',
      price: 45000000, // $450,000
      wholesalePrice: 42000000,
      description: 'New Holland CX8090 with 9m header. Yield mapping and GPS included.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      wholesaleAvailable: true,
      wholesaleListedAt: new Date()
    },
    {
      dealerId: createdAgDealers[1].id,
      type: 'Tractor',
      brand: 'Massey Ferguson',
      model: '8S.265',
      year: 2023,
      serialNumber: 'MF8S265HJL901234',
      horsepower: 265,
      engineType: 'Diesel',
      hours: 450,
      ptoType: '540/1000',
      hydraulicFlow: 190,
      liftCapacity: 10000,
      condition: 'New',
      price: 38500000, // $385,000
      description: 'Brand new Massey Ferguson 8S.265 Dyna-VT. Full manufacturer warranty.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced
    },
    {
      dealerId: createdAgDealers[1].id,
      type: 'Cultivation',
      brand: 'Kverneland',
      model: 'CLC Pro 450',
      year: 2022,
      width: 4500,
      weight: 3800,
      condition: 'Excellent',
      price: 4500000, // $45,000
      description: 'Kverneland CLC Pro cultivator with crumbler roller. 4.5m working width.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.none
    },
    // Southern Farm Equipment
    {
      dealerId: createdAgDealers[2].id,
      type: 'Tractor',
      brand: 'Case IH',
      model: 'Magnum 340',
      year: 2021,
      serialNumber: 'CIHM340HJK234567',
      horsepower: 340,
      engineType: 'Diesel',
      hours: 2200,
      ptoType: '1000',
      hydraulicFlow: 227,
      liftCapacity: 8000,
      condition: 'Good',
      price: 29500000, // $295,000
      wholesalePrice: 27500000,
      description: 'Case IH Magnum 340 CVXDrive with full guidance. Ideal for heavy cultivation.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced,
      wholesaleAvailable: true,
      wholesaleListedAt: new Date()
    },
    {
      dealerId: createdAgDealers[2].id,
      type: 'Grass Machinery',
      brand: 'Krone',
      model: 'BiG M 450',
      year: 2020,
      serialNumber: 'KBM450HJK890123',
      horsepower: 530,
      engineType: 'Diesel',
      hours: 1200,
      width: 9500,
      condition: 'Excellent',
      price: 52000000, // $520,000
      description: 'Krone BiG M 450 self-propelled mower. Triple mower conditioner configuration.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.synced
    },
    {
      dealerId: createdAgDealers[2].id,
      type: 'Feed Mixer',
      brand: 'Kuhn',
      model: 'SPV 14.1',
      year: 2022,
      weight: 3500,
      condition: 'Excellent',
      price: 7500000, // $75,000
      description: 'Kuhn SPV 14.1 self-propelled mixer wagon. 14 cubic metre capacity.',
      status: VehicleStatus.reserved,
      syndicationStatus: SyndicationStatus.none
    },
    {
      dealerId: createdAgDealers[2].id,
      type: 'ATV/UTV',
      brand: 'Polaris',
      model: 'Ranger XP 1000',
      year: 2023,
      horsepower: 82,
      engineType: 'Petrol',
      hours: 120,
      condition: 'Excellent',
      price: 3500000, // $35,000
      description: 'Polaris Ranger XP 1000 NorthStar with heating and air conditioning.',
      status: VehicleStatus.in_stock,
      syndicationStatus: SyndicationStatus.none
    }
  ]

  for (const item of equipment) {
    await prisma.equipment.create({
      data: item
    })
  }
  console.log('Created', equipment.length, 'sample equipment items')

  console.log('\nSeed completed successfully!')
  console.log('\nTest Accounts:')
  console.log('  Platform Admin: admin@dms.co.nz / admin123')
  console.log('')
  console.log('  VEHICLES SECTOR:')
  console.log('  Group Admin (Tristram Group): admin@tristramgroup.co.nz / password123')
  console.log('  Dealer Admin (Auckland Motors): john@aucklandmotors.co.nz / password123')
  console.log('  Dealer User (Auckland Motors): sarah@aucklandmotors.co.nz / password123')
  console.log('  Dealer Admin (Wellington): mike@wellingtonauto.co.nz / password123')
  console.log('  Dealer Admin (Christchurch): lisa@chchcars.co.nz / password123')
  console.log('  Dealer Admin (Tristram Auckland): sales@tristramauckland.co.nz / password123')
  console.log('')
  console.log('  AGRICULTURE SECTOR:')
  console.log('  Dealer Admin (Power Farming Auckland): tom@powerfarming-auckland.co.nz / password123')
  console.log('  Dealer Admin (Central Tractors): jim@centraltractors.co.nz / password123')
  console.log('  Dealer Admin (Southern Farm): bob@southernfarm.co.nz / password123')
}

main()
  .catch((e) => {
    console.error(e)
    process.exit(1)
  })
  .finally(async () => {
    await prisma.$disconnect()
  })
